#include <jsc/client.hpp>

#include <chrono>
#include <cstdlib> // For getenv
#include <exception>
#include <string>
#include <thread>
#include <unordered_map>
#include <utility> // For move

#include <jsc/client_key.hpp>
#include <jsc/detail/attestation.hpp>
#include <jsc/detail/encoding.hpp>
#include <jsc/error.hpp>
#include <jsc/logger.hpp>
#include <jsc/vbs/host.hpp>

namespace jsc {

using namespace detail;

static Log log_from_config(ClientConfig const &config) {
  std::string log_dir = config.log_dir;
  if (log_dir.empty()) {
    auto env_log_dir = std::getenv("LOGDIR");
    if (env_log_dir != nullptr) {
      log_dir = env_log_dir;
    }
  }
  std::string logger_name = "jsc_client.log";
  std::string log_file_path =
      log_dir.empty() ? "" : log_dir + "/" + logger_name;
  return {logger_name, log_file_path};
}

Client::Client(ClientConfig const &config)
    : config{config}, logger{log_from_config(config)} {

  // Create and initialize VBS enclave if available
  Enclave enclave;
  if (config.vbs_preference >= 2 ||
      (config.vbs_preference == 1 && detail::is_vbs_supported())) {
    try {
      enclave =
          Enclave::create(L"jsc_crypto_vbs.dll", /*is_vbs=*/true, &logger);
    } catch (std::exception const &e) {
      logger.warn("VBS enclave creation failed: {}", e.what());
      enclave =
          Enclave::create(L"jsc_crypto_host.dll", /*is_vbs=*/false, &logger);
    }
  } else {
    enclave =
        Enclave::create(L"jsc_crypto_host.dll", /*is_vbs=*/false, &logger);
  }
  key_store = {std::move(enclave)};

  // Parse root_key
  if (!config.root_key_config.empty()) {
    root_key_parse();
  }

  // Remote attestation
  attest_server_no_throw();

  // Start RA timer
  if (config.attest_interval > 0) {
    auto interval = config.attest_interval;
    attest_timer = std::thread{[this, interval]() {
      while (true) {
        {
          std::unique_lock<std::mutex> lock{attest_mutex};
          if (attest_cv.wait_for(lock, std::chrono::seconds(interval),
                                 [this]() { return attest_stop; })) {
            break;
          }
        }
        attest_server_no_throw();
      }
    }};
  }
}

Client::~Client() noexcept {
  try {
    attest_stop = true;
    {
      std::unique_lock<std::mutex> lock{attest_mutex};
      attest_cv.notify_all();
    }
    if (attest_timer.joinable()) {
      attest_timer.join();
    }
  } catch (...) {
    // Ignored
  }
}

void Client::root_key_parse() {
  try {
    for (auto const &[app_info, private_pem] : config.root_key_config) {
      auto root_key = key_store.load_rsa_private_key(private_pem);
      root_key_info.emplace(app_info, std::move(root_key));
    }

  } catch (std::exception const &e) {
    logger.error("root_key_parse failed: {}", e.what());
    throw ConfigError{"root_key_config", e.what()};
  }
}

void Client::attest_server_no_throw() noexcept {
  try {
    attest_server();
  } catch (std::exception const &e) {
    logger.error("Failed to attest server: {}", e.what());
  } catch (...) {
    logger.error("Failed to attest server");
  }
}

bool Client::attest_server() {
  static constexpr unsigned kMaxTryCount = 3;

  logger.info("attest_server active");
  for (unsigned count = 0; count < kMaxTryCount; ++count) {
    auto [success, public_key] = do_attestation(config, logger);
    if (success) {
      logger.info("ra attest success, pub_key={}", public_key);

      std::unique_lock<std::mutex> lock{session_key_mutex};
      session_key = key_store.load_rsa_public_key(public_key);
      return true;
    } else {
      logger.error("ra attest failed, try again...");
    }
  }
  return false;
}

EncryptResult Client::encrypt_with_response(std::string_view plaintext) {
  std::unique_lock<std::mutex> lock{session_key_mutex};
  return session_key.encrypt(plaintext);
}

EncryptResult Client::encrypt_with_response(std::istream &source,
                                            std::ostream &dest) {
  std::unique_lock<std::mutex> lock{session_key_mutex};
  return session_key.encrypt(source, dest);
}

std::string Client::gen_sign(std::string const &app_info,
                             std::string_view message) {
  if (app_info.empty())
    throw ParamError{"app_info", "empty"};
  if (message.empty())
    throw ParamError{"message", "empty"};

  auto it = root_key_info.find(app_info);
  if (it == root_key_info.end()) {
    logger.error(
        "gen_sign error, can not find app_info in root key conf, app_info={}",
        app_info);
    throw ParamError{"app_info", "not found in configuration"};
  }

  auto signature = it->second.sign(message);
  return base64_encode(signature);
}

} // namespace jsc
