#pragma once

#include <condition_variable>
#include <fstream>
#include <istream>
#include <mutex>
#include <ostream>
#include <string>
#include <string_view>
#include <thread>
#include <type_traits>
#include <unordered_map>
#include <utility> // For forward

#include <jsc/client_config.hpp> // IWYU pragma: export
#include <jsc/client_key.hpp>    // IWYU pragma: export
#include <jsc/logger.hpp>
#include <jsc/vbs/host_key.hpp>

namespace jsc {

class Client {
private:
  ClientConfig config;

  Log logger;

  detail::EnclaveKeyStore key_store;

  std::mutex session_key_mutex;
  detail::SessionKey session_key;

  std::unordered_map<std::string, detail::SigningKey> root_key_info;

  std::thread attest_timer;
  std::condition_variable attest_cv;
  std::mutex attest_mutex;
  bool attest_stop;

  void root_key_parse();

  void attest_server_no_throw() noexcept;

public:
  __declspec(dllexport) explicit Client(ClientConfig const &config);

  __declspec(dllexport) ~Client() noexcept;

  __declspec(dllexport) bool attest_server();

  Log &log() noexcept { return logger; }

  std::string encrypt(std::string_view data) {
    return encrypt_with_response(data).ciphertext;
  }

  __declspec(dllexport) EncryptResult
  encrypt_with_response(std::string_view data);

  std::string encrypt(std::istream &source, std::ostream &dest) {
    return encrypt_with_response(source, dest).ciphertext;
  }

  __declspec(dllexport) EncryptResult
  encrypt_with_response(std::istream &source, std::ostream &dest);

  // Path can be: string const &, wstring const &, char const *, wchar_t const *
  template <typename Path>
  std::string encrypt_file(Path source_path, Path dest_path) {
    return encrypt_file_with_response(std::forward<Path>(source_path),
                                      std::forward<Path>(dest_path))
        .ciphertext;
  }

  template <typename Path>
  EncryptResult encrypt_file_with_response(Path source_path, Path dest_path) {
    std::ifstream source{std::forward<Path>(source_path), std::ios::binary};
    if (!source)
      throw std::runtime_error{"Cannot open source"};

    std::ofstream dest{std::forward<Path>(dest_path), std::ios::binary};
    if (!source)
      throw std::runtime_error{"Cannot open dest"};

    return encrypt_with_response(source, dest);
  }

  __declspec(dllexport) std::string gen_sign(std::string const &app_info,
                                             std::string_view message);
};

// 不可默认构造
static_assert(not std::is_default_constructible_v<Client>);
// 可以析构
static_assert(std::is_nothrow_destructible_v<Client>);
// 不可复制
static_assert(not std::is_copy_constructible_v<Client> and
              not std::is_copy_assignable_v<Client>);
// 不可移动
static_assert(not std::is_move_constructible_v<Client> and
              not std::is_move_assignable_v<Client> and
              not std::is_swappable_v<Client>);

} // namespace jsc
