#pragma once

#include <cstddef> // For size_t
#include <fstream>
#include <istream>
#include <ostream>
#include <string>
#include <string_view>
#include <type_traits>
#include <utility> // For forward, move
#include <vector>

#include <jsc/vbs/host_key.hpp>

namespace jsc {

// Forward declarations
class Client;
namespace detail {
class SessionKey;
}

class ResponseKey {
private:
  detail::DecryptingKey key;

  ResponseKey(detail::DecryptingKey &&key) noexcept : key{std::move(key)} {}

  friend detail::SessionKey;

public:
  ResponseKey() noexcept = default;

  __declspec(dllexport) std::vector<uint8_t> decrypt(std::string_view response);

  __declspec(dllexport) void decrypt(std::string_view metadata,
                                     std::istream &source, std::ostream &dest);

  template <typename Path>
  void decrypt_file(std::string_view metadata, Path source_path,
                    Path dest_path) {
    std::ifstream source{std::forward<Path>(source_path), std::ios::binary};
    if (!source)
      throw std::runtime_error{"Cannot open source"};

    std::ofstream dest{std::forward<Path>(dest_path), std::ios::binary};
    if (!source)
      throw std::runtime_error{"Cannot open dest"};

    decrypt(metadata, source, dest);
  }
};

// 可以默认构造
static_assert(std::is_nothrow_default_constructible_v<ResponseKey>);
// 可以析构
static_assert(std::is_nothrow_destructible_v<ResponseKey>);
// 不可复制
static_assert(not std::is_copy_constructible_v<ResponseKey> and
              not std::is_copy_assignable_v<ResponseKey>);
// 可以移动
static_assert(std::is_nothrow_move_constructible_v<ResponseKey> and
              std::is_nothrow_move_assignable_v<ResponseKey> and
              std::is_nothrow_swappable_v<ResponseKey>);

struct EncryptResult {
  std::string ciphertext;

  ResponseKey response_key;
};

namespace detail {

class SessionKey {
private:
  EncryptingKey key;

  SessionKey(EncryptingKey &&key) noexcept : key{std::move(key)} {}

  friend Client;

public:
  SessionKey() noexcept = default;

  EncryptResult encrypt(uint8_t const *data, size_t len);
  EncryptResult encrypt(std::string_view data);

  EncryptResult encrypt(std::istream &source, std::ostream &dest);
};

// 可以默认构造
static_assert(std::is_nothrow_default_constructible_v<SessionKey>);
// 可以析构
static_assert(std::is_nothrow_destructible_v<SessionKey>);
// 不可复制
static_assert(not std::is_copy_constructible_v<SessionKey> and
              not std::is_copy_assignable_v<SessionKey>);
// 可以移动
static_assert(std::is_nothrow_move_constructible_v<SessionKey> and
              std::is_nothrow_move_assignable_v<SessionKey> and
              std::is_nothrow_swappable_v<SessionKey>);

} // namespace detail

} // namespace jsc
