#include <jsc/detail/attestation.hpp>

#include <string>
#include <string_view>
#include <utility> // For move

#include <nlohmann/json.hpp>

#include <jsc/client_config.hpp>
#include <jsc/detail/bytedance_gateway.hpp>
#include <jsc/detail/encoding.hpp>
#include <jsc/error.hpp>
#include <jsc/logger.hpp>

namespace jsc {
namespace detail {

httplib::Result request_tca(ClientConfig const &config, Log &logger) {
  TopInfo top_info;
  try {
    top_info =
        TopInfo::from_json(nlohmann::json::parse(config.bytedance_top_info));
  } catch (std::exception const &e) {
    logger.error("Invalid 'bytedance_top_info' configuration: {}", e.what());
    throw ConfigError{"bytedance_top_info", e.what()};
  }

  httplib::Headers headers{
      {"UID", config.ra_uid},
  };
  nlohmann::json body{
      {"PolicyID", config.ra_policy_id},
      {"KeyNegotiation", config.ra_key_negotiation},
      {"Token", true},
  };
  if (!config.ra_service_name.empty()) {
    body["ServiceName"] = config.ra_service_name;
  } else if (!config.ra_pods_info.empty()) {
    body["AttestedPods"] =
        nlohmann::json::array({nlohmann::json::parse(config.ra_pods_info)});
  } else {
    throw ConfigError{"bytedance_top_info", "Service or pod not specified"};
  }

  return request_bytedance_gateway(top_info, "/", body.dump(),
                                   std::move(headers));
}

[[nodiscard]]
static bool check_ra(nlohmann::json const &ra_info, Log &logger) {
  if (!ra_info.contains("evidence") || !ra_info.contains("token")) {
    logger.warn("No RA evidence");
    return false;
  }
  try {
    auto raw_evidence = ra_info.at("evidence").get<std::string_view>();
    auto raw_evidence_formatted = nlohmann::json::parse(raw_evidence).dump(4);
    logger.debug("RA evidence: {}", raw_evidence_formatted);

    auto raw_token = ra_info.at("token").get<std::string_view>();
    auto first_dot_pos = raw_token.find('.'),
         second_dot_pos = raw_token.find('.', first_dot_pos + 1);
    if (second_dot_pos == raw_token.npos) {
      logger.error("Bad RA token: {}", raw_token);
      return false;
    }
    auto token_data = base64_decode_string(
        raw_token.substr(first_dot_pos + 1, second_dot_pos - first_dot_pos));
    logger.debug("RA token data: {}", token_data);

    auto token_object = nlohmann::json::parse(std::move(token_data));
    auto policy_ok = token_object.at("policies_matched").size() > 0;
    return policy_ok;

  } catch (std::exception const &e) {
    logger.error("Check RA error: {}", e.what());
    return false;
  }
}

std::pair<bool, std::string> do_attestation(ClientConfig const &config,
                                            Log &logger) {
  auto response = request_tca(config, logger);

  if (!response) {
    auto message = httplib::to_string(response.error());
    logger.error("RA error: url={} error={}", config.ra_url, message);
    throw ServiceError{"RA", message};
  }

  if (response->status != 200) {
    logger.error("Service error: service={} code={} body={}", config.ra_url,
                 response->status, response->body);
    throw ServiceError{"RA",
                       "response status " + std::to_string(response->status)};
  }

  nlohmann::json ra_res;
  try {
    ra_res = nlohmann::json::parse(response->body).at("Result");
  } catch (std::exception const &e) {
    logger.error("Response is not JSON: service={} error={} response={}",
                 config.ra_url, e.what(), response->body);
    throw ServiceError{"RA", "invalid JSON in response"};
  }
  if (!ra_res.is_object() || ra_res.size() == 0) {
    logger.error("attest_server failed, Result is empty, ra response={}",
                 response->body);
    return {false, ""};
  }

  for (auto const &[cluster_info, ra_info] : ra_res.items()) {
    if (check_ra(ra_info, logger)) {
      logger.info("RA success");
    }

    auto public_key =
        ra_info.at("key_info").at("pub_key_info").get<std::string_view>();
    return {true, std::string{public_key}};
  }

  return {false, ""};
}

} // namespace detail
} // namespace jsc
