#include <jsc/detail/bytedance_gateway.hpp>

#include <cstddef> // For size_t
#include <cstdint>
#include <ctime>   // For time, tm, gmtime
#include <iomanip> // For put_time
#include <map>
#include <sstream>
#include <string_view>
#include <utility> // For move

#include <nlohmann/json.hpp>
#include <openssl/sha.h>

#include <jsc/detail/encoding.hpp>
#include <jsc/detail/openssl_utils.hpp>

namespace jsc {
namespace detail {

TopInfo TopInfo::from_json(nlohmann::json const &obj) {
  return {
      .ak = obj.at("ak").get<std::string>(),
      .sk = obj.at("sk").get<std::string>(),
      .service = obj.at("service").get<std::string>(),
      .region = obj.value("region", "cn-beijing"),
      .method = obj.value("method", "POST"),
      .action = obj.value("action", "GetAttestationBackend"),
      .version = obj.value("version", "2024-12-24"),

      .host = obj.at("url").get<std::string>(),
      .url_rewrite = obj.value("url_rewrite", ""),
  };
}

httplib::Params build_top_params(TopInfo const &top_info) {
  return {
      {"Action", top_info.action},
      {"Version", top_info.version},
  };
}

static std::vector<uint8_t> hmac_sha256(uint8_t const *key, size_t key_len,
                                        std::string const &content) {
  std::vector<uint8_t> digest(SHA256_DIGEST_LENGTH);
  unsigned digest_len;

  if (!HMAC(EVP_sha256(), key, static_cast<int32_t>(key_len),
            reinterpret_cast<uint8_t const *>(content.data()), content.size(),
            digest.data(), &digest_len))
    throw openssl_error("HMAC");

  return digest;
}

template <typename T>
static std::vector<uint8_t> hmac_sha256(T const &key,
                                        std::string const &content) {
  return hmac_sha256(reinterpret_cast<uint8_t const *>(key.data()), key.size(),
                     content);
}

static std::string hash_sha256(std::string const &data) {
  std::vector<uint8_t> digest(SHA256_DIGEST_LENGTH);
  if (!SHA256(reinterpret_cast<uint8_t const *>(data.data()), data.size(),
              digest.data()))
    throw openssl_error("SHA256");

  return hex_encode(digest);
}

static std::string to_lowercase(std::string s) noexcept {
  for (char &c : s) {
    if (c >= 'A' && c <= 'Z') {
      c -= 'A' - 'a';
    }
  }
  return s;
}

httplib::Headers build_top_headers(TopInfo const &top_info,
                                   std::string const &path,
                                   std::string const &body) {
  std::string now_date, now_date_time;
  {
    auto now_timestamp = std::time(nullptr);
    auto now = *gmtime(&now_timestamp);
    std::stringstream ss;
    ss << std::put_time(&now, "%Y%m%dT%H%M%SZ");
    now_date_time = ss.str();
    now_date = now_date_time.substr(0, 8);
  }

  auto content_sha256 = hash_sha256(body);

  std::map<std::string, std::string> headers{
      {"Content-Type", "application/json"},
      {"Host", top_info.host},
      {"X-Content-Sha256", content_sha256},
      {"X-Date", now_date_time},
  };

  std::string signed_headers;
  char const *sep{""};
  for (auto const &[key, value] : headers) {
    signed_headers += sep + to_lowercase(key);
    sep = ";";
  }

  auto canonical_request = top_info.method + '\n';
  auto path_query_pos = path.find('?');
  if (path_query_pos == path.npos) {
    canonical_request += path + "\n\n";
  } else {
    canonical_request += path.substr(0, path_query_pos) + '\n';
    canonical_request += path.substr(path_query_pos + 1) + '\n';
  }
  for (auto const &[key, value] : headers) {
    canonical_request += to_lowercase(key) + ':' + value + '\n';
  }
  canonical_request += '\n' + signed_headers + '\n' + content_sha256;

  auto canonical_request_hash = hash_sha256(canonical_request);

  auto credential_scope =
      now_date + '/' + top_info.region + '/' + top_info.service + "/request";
  auto string_to_sign = "HMAC-SHA256\n" + now_date_time + '\n' +
                        credential_scope + '\n' + canonical_request_hash;

  auto signature = hmac_sha256(top_info.sk, now_date);
  signature = hmac_sha256(signature, top_info.region);
  signature = hmac_sha256(signature, top_info.service);
  signature = hmac_sha256(signature, "request");
  signature = hmac_sha256(signature, string_to_sign);
  auto signature_hex = hex_encode(signature);

  headers.emplace("Authorization", "HMAC-SHA256 Credential=" + top_info.ak +
                                       '/' + credential_scope +
                                       ", SignedHeaders=" + signed_headers +
                                       ", Signature=" + signature_hex);

  // Copy to unordered_map
  return {headers.cbegin(), headers.cend()};
}

namespace {
struct ParsedUrl {
  std::string_view host;
  std::string_view path;

  static ParsedUrl parse(std::string_view url) {
    constexpr std::string_view kHttpsScheme = "https://";
    if (url.substr(0, kHttpsScheme.size()) == kHttpsScheme) {
      url.remove_prefix(kHttpsScheme.size());
    }
    auto slash_pos = url.find('/');
    if (slash_pos == url.npos) {
      return {.host{url}, .path{}};
    }
    return {.host{url.substr(0, slash_pos)}, .path{url.substr(slash_pos)}};
  }
};
} // namespace

httplib::Result
request_bytedance_gateway(TopInfo const &top_info, std::string const &path,
                          std::string &&body,
                          httplib::Headers &&additional_headers) {
  httplib::Request req;
  auto params = build_top_params(top_info);
  req.method = top_info.method;
  req.path = httplib::append_query_params(path, params);
  req.headers = build_top_headers(top_info, req.path, body);
  req.headers.merge(std::move(additional_headers));
  req.body = std::move(body);

  std::string host_rewrite;
  if (top_info.url_rewrite.empty()) {
    host_rewrite = top_info.host;
  } else {
    auto parsed = ParsedUrl::parse(top_info.url_rewrite);
    host_rewrite = parsed.host;
    req.path = httplib::append_query_params(std::string{parsed.path}, params);
    req.headers.find("Host")->second = parsed.host;
  }

  httplib::SSLClient cli{host_rewrite};

  /*cli.load_ca_cert_store();*/
  cli.enable_server_certificate_verification(false);

  return cli.send(req);
}

} // namespace detail
} // namespace jsc
