#pragma once

#include <cstddef> // For size_t
#include <string_view>
#include <tuple>
#include <vector>

#include <jsc/detail/consts.hpp> // IWYU pragma: export

namespace jsc {
namespace detail {

template <typename Impl>
struct cipher_impl {
  using rsa_key = typename Impl::rsa_key;
  using sym_key = typename Impl::sym_key;
  using encrypt_ctx = typename Impl::encrypt_ctx;
  using decrypt_ctx = typename Impl::decrypt_ctx;

  static rsa_key parse_rsa_public_key(std::string_view pem);

  static rsa_key parse_rsa_private_key(std::string_view blob);

  static std::vector<uint8_t> rsa_encrypt(rsa_key &key, uint8_t const *data,
                                          size_t len);

  static std::vector<uint8_t> rsa_decrypt(rsa_key &key, uint8_t const *data,
                                          size_t len);

  static std::vector<uint8_t> rsa_sign(rsa_key &key, uint8_t const *data,
                                       size_t len);

  static sym_key generate_sym_key();

  /** @returns encryptor, iv */
  static std::tuple<encrypt_ctx, std::vector<uint8_t>>
  encrypt_init(sym_key &secret);

  /** @returns ciphertext */
  static std::vector<uint8_t>
  encrypt_update(encrypt_ctx &ctx, uint8_t const *plaintext, size_t len);

  /** @returns remaining ciphertext, tag */
  [[nodiscard]]
  static std::tuple<std::vector<uint8_t>, std::vector<uint8_t>>
  encrypt_finish(encrypt_ctx &ctx);

  static decrypt_ctx decrypt_init(sym_key &secret, uint8_t const *iv,
                                  uint8_t const *tag);

  /** @returns plaintext */
  static std::vector<uint8_t>
  decrypt_update(decrypt_ctx &ctx, uint8_t const *ciphertext, size_t len);

  /** @returns remaining plaintext */
  [[nodiscard]]
  static std::vector<uint8_t> decrypt_finish(decrypt_ctx &ctx);
};

} // namespace detail
} // namespace jsc
