#include <jsc/logger.hpp>

#include <iostream>
#include <memory> // For make_shared
#include <string>
#include <utility> // For move
#include <vector>

#include <spdlog/sinks/basic_file_sink.h>
#include <spdlog/sinks/stdout_color_sinks.h>
#include <spdlog/spdlog.h>

namespace jsc {

Log::Log(std::string name, std::string const &file_path)
    : name{std::move(name)} {
  try {
    // 控制台输出
    auto console_sink = std::make_shared<spdlog::sinks::stdout_color_sink_mt>();

    std::vector<spdlog::sink_ptr> sinks{std::move(console_sink)};

    // 文件输出
    if (!file_path.empty()) {
      auto file_sink =
          std::make_shared<spdlog::sinks::basic_file_sink_mt>(file_path, true);
      sinks.emplace_back(std::move(file_sink));
    }

    // 创建 logger
    logger = std::make_shared<spdlog::logger>(this->name, sinks.begin(),
                                              sinks.end());

    // 设置日志格式
    logger->set_pattern("[%Y-%m-%d %H:%M:%S] [%l] %v");

    // 设置日志级别
    logger->set_level(spdlog::level::info);

    // 注册 logger
    spdlog::register_logger(logger);

  } catch (spdlog::spdlog_ex const &ex) {
    std::cerr << "Log initialization failed: " << ex.what() << std::endl;
  }
}

Log::~Log() { spdlog::drop(name); }

void Log::set_level(spdlog::level::level_enum level) {
  if (logger) {
    logger->set_level(level);
  }
}

} // namespace jsc
