#pragma once

#include <memory> // For shared_ptr
#include <string>
#include <type_traits>
#include <utility> // For forward

#include <spdlog/logger.h>

namespace jsc {

class Log {
private:
  std::string name;

  std::shared_ptr<spdlog::logger> logger;

public:
  /// 构造日志工具类, 将日志写入指定文件, 并注册到 spdlog.
  /// 若 log_path 为空, 则不会写入文件.
  __declspec(dllexport) Log(std::string name, std::string const &file_path);

  /// 将日志工具的实例从 spdlog 取消注册.
  __declspec(dllexport) ~Log();

  Log(Log const &) = delete;
  Log &operator=(Log const &) = delete;
  Log(Log &&) noexcept = default;
  Log &operator=(Log &&) noexcept = default;

  /// 记录调试日志.
  void debug(spdlog::string_view_t msg) {
    if (logger) {
      logger->log(spdlog::level::debug, msg);
    }
  }

  /// 记录调试日志.
  template <typename... Args>
  void debug(spdlog::format_string_t<Args...> fmt, Args &&...args) {
    if (logger) {
      logger->log(spdlog::level::debug, fmt, std::forward<Args>(args)...);
    }
  }

  /// 记录信息日志.
  void info(spdlog::string_view_t msg) {
    if (logger) {
      logger->log(spdlog::level::info, msg);
    }
  }

  /// 记录信息日志.
  template <typename... Args>
  void info(spdlog::format_string_t<Args...> fmt, Args &&...args) {
    if (logger) {
      logger->log(spdlog::level::info, fmt, std::forward<Args>(args)...);
    }
  }

  /// 记录警告日志.
  void warn(spdlog::string_view_t msg) {
    if (logger) {
      logger->log(spdlog::level::warn, msg);
    }
  }

  /// 记录警告日志.
  template <typename... Args>
  void warn(spdlog::format_string_t<Args...> fmt, Args &&...args) {
    if (logger) {
      logger->log(spdlog::level::warn, fmt, std::forward<Args>(args)...);
    }
  }

  /// 记录错误日志.
  void error(spdlog::string_view_t msg) {
    if (logger) {
      logger->log(spdlog::level::err, msg);
    }
  }

  /// 记录错误日志.
  template <typename... Args>
  void error(spdlog::format_string_t<Args...> fmt, Args &&...args) {
    if (logger) {
      logger->log(spdlog::level::err, fmt, std::forward<Args>(args)...);
    }
  }

  /// 设置日志级别.
  __declspec(dllexport) void set_level(spdlog::level::level_enum level);
};

// 不可默认构造
static_assert(not std::is_default_constructible_v<Log>);
// 可以析构
static_assert(std::is_nothrow_destructible_v<Log>);
// 不可复制
static_assert(not std::is_copy_constructible_v<Log> and
              not std::is_copy_assignable_v<Log>);
// 可以移动
static_assert(std::is_nothrow_move_constructible_v<Log> and
              std::is_nothrow_move_assignable_v<Log> and
              std::is_nothrow_swappable_v<Log>);

} // namespace jsc
