// Avoid error "Cannot include both windows.h and winenclave.h"
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <winenclave.h>

#include <jsc/vbs/host.hpp> // IWYU pragma: associated

#include <cstdint>
#include <stdexcept>
#include <string>
#include <string_view>

#include <jsc/error.hpp>
#include <jsc/logger.hpp>
#include <jsc/vbs/interface.hpp>

namespace jsc {
namespace detail {

static std::runtime_error win_error(std::string_view origin) {
  return std::runtime_error{std::string{origin} + ": Windows error " +
                            std::to_string(GetLastError())};
}

bool is_vbs_supported() { return IsEnclaveTypeSupported(ENCLAVE_TYPE_VBS); }

static constexpr int kCreateFlags =
#ifdef NDEBUG
    0;
#else
    ENCLAVE_VBS_FLAG_DEBUG;
#endif

Enclave Enclave::create(wchar_t const *dll_path, bool is_vbs, Log *logger) {
  if (is_vbs) {
    // Create
    constexpr ENCLAVE_CREATE_INFO_VBS create_info{
        /*Flags=*/kCreateFlags,
        /*OwnerID=*/{0x10, 0x20, 0x30, 0x40, 0x41, 0x31, 0x21, 0x11},
    };
    auto handle = CreateEnclave(GetCurrentProcess(),
                                /*address=*/nullptr,
                                /*size=*/0x10000000,
                                /*initialCommitment=*/0, ENCLAVE_TYPE_VBS,
                                &create_info, sizeof(ENCLAVE_CREATE_INFO_VBS),
                                /*enclaveError=*/nullptr);
    if (!handle)
      throw win_error("CreateEnclave");

    Enclave enclave{handle, is_vbs, logger};
    if (logger)
      logger->info("Created enclave {}", enclave.handle);
    // Enclave will be cleaned-up if error is thrown during initalization

    // Initalize
    if (!LoadEnclaveImageW(handle, dll_path))
      throw win_error("LoadEnclaveImage");

    constexpr ENCLAVE_INIT_INFO_VBS init_info{
        /*Length=*/sizeof(ENCLAVE_INIT_INFO_VBS),
        /*ThreadCount=*/1,
    };
    if (!InitializeEnclave(GetCurrentProcess(), handle, &init_info,
                           init_info.Length, /*enclaveError=*/nullptr))
      throw win_error("InitializeEnclave");

    return enclave;

  } else {
    auto handle = LoadLibraryW(dll_path);
    if (!handle)
      throw win_error("LoadLibrary");

    if (logger)
      logger->info("Loaded library");

    return {handle, is_vbs, logger};
  }
}

Enclave::~Enclave() noexcept {
  if (!handle)
    return;

  if (is_vbs) {
    if (logger)
      logger->info("Destroying enclave {}", handle);
    if (!TerminateEnclave(handle, /*wait=*/TRUE)) {
      if (logger)
        logger->error("TerminateEnclave: {}", GetLastError());
    }
    if (!DeleteEnclave(handle)) {
      if (logger)
        logger->error("DeleteEnclave: {}", GetLastError());
    }
  } else {
    if (logger)
      logger->info("Unloading library");
    if (!FreeLibrary(reinterpret_cast<HMODULE>(handle))) {
      if (logger)
        logger->error("FreeLibrary: {}", GetLastError());
    }
  }
}

uintptr_t Enclave::invoke(vbs::routine_id_t id, char const *name, void *args) {
  if (logger)
    logger->debug("Enclave call {}", name);
  if (!handle)
    throw StateError{"no enclave"};

  void *routine = routines[id];
  if (!routine) {
    routine = GetProcAddress(reinterpret_cast<HMODULE>(handle), name);
    if (!routine)
      throw win_error(std::string{"GetProcAddress("} + name + ")");
    routines[id] = routine;
  }

  uintptr_t result{};
  if (is_vbs) {
    if (!CallEnclave(reinterpret_cast<PENCLAVE_ROUTINE>(routine), args,
                     /*waitForThread=*/TRUE,
                     reinterpret_cast<void **>(&result)))
      throw win_error(std::string{"CallEnclave("} + name + ")");
  } else {
    result = reinterpret_cast<uintptr_t (*)(void *)>(routine)(args);
  }
  return result;
}

} // namespace detail
} // namespace jsc
