#pragma once

#include <array>
#include <cstdint>
#include <string_view>
#include <type_traits>
#include <utility> // For forward, move, swap
#include <vector>

#include <jsc/error.hpp>
#include <jsc/logger.hpp>
#include <jsc/vbs/interface.hpp>

namespace jsc {
namespace detail {

using enclave_handle_t = void *;

bool is_vbs_supported();

class Enclave {
private:
  enclave_handle_t handle;

  bool is_vbs;

  Log *logger;

  std::array<void *, vbs::kRoutineCount> routines;

  constexpr Enclave(enclave_handle_t handle, bool is_vbs, Log *logger) noexcept
      : handle{handle}, is_vbs{is_vbs}, logger{logger}, routines{} {}

  uintptr_t invoke(vbs::routine_id_t id, char const *name, void *args);

public:
  // Creates an Enclave handle that is not bound to any enclave.
  constexpr Enclave() noexcept : handle{}, is_vbs{}, logger{}, routines{} {}

  // Destroys the enclave that this handle is bound to.
  ~Enclave() noexcept;

  friend constexpr void swap(Enclave &first, Enclave &second) noexcept {
    using std::swap;
    swap(first.handle, second.handle);
    swap(first.is_vbs, second.is_vbs);
    swap(first.logger, second.logger);
    swap(first.routines, second.routines);
  }

  constexpr Enclave(Enclave &&other) noexcept : Enclave{} {
    swap(*this, other);
  }

  constexpr Enclave &operator=(Enclave &&other) noexcept {
    swap(*this, other);
    return *this;
  }

  // Creates an Enclave handle that is bound to a new enclave.
  static Enclave create(wchar_t const *dll_path, bool is_vbs, Log *logger);

  template <typename Routine, typename... Args>
  routine_result_t<Routine> call(Args &&...args) {
    size_t error_len = 100;
    std::vector<char> error(error_len);
    routine_args_with_error_t<Routine> args_tuple{
        {std::forward<Args>(args)...},
        error.data(),
        &error_len,
    };
    auto result = invoke(Routine::id, Routine::name, &args_tuple);
    if (result == static_cast<uintptr_t>(-1)) {
      throw CryptoError{{error.data(), error_len}};
    }
    if constexpr (std::is_void_v<routine_result_t<Routine>>) {
      return;
    } else {
      return static_cast<routine_result_t<Routine>>(result);
    }
  }
};

// 可以默认构造
static_assert(std::is_nothrow_default_constructible_v<Enclave>);
// 可以析构
static_assert(std::is_nothrow_destructible_v<Enclave>);
// 不可复制
static_assert(not std::is_copy_constructible_v<Enclave> and
              not std::is_copy_assignable_v<Enclave>);
// 可以移动
static_assert(std::is_nothrow_move_constructible_v<Enclave> and
              std::is_nothrow_move_assignable_v<Enclave> and
              std::is_nothrow_swappable_v<Enclave>);

} // namespace detail
} // namespace jsc
