#pragma once

#include <cstddef> // For size_t
#include <cstdint>
#include <string_view>
#include <type_traits>
#include <utility> // For move, swap
#include <vector>

#include <jsc/vbs/host.hpp>
#include <jsc/vbs/interface.hpp>

namespace jsc {
namespace detail {

// Forward declarations
class DecryptingKey;
class EnclaveKeyStore;
class EncryptingKey;

class EnclaveObject {
protected:
  Enclave *enclave;
  vbs::object_id_t id;

  constexpr EnclaveObject(Enclave *enclave, vbs::object_id_t id) noexcept
      : enclave{enclave}, id{id} {}

  friend EnclaveKeyStore;

public:
  constexpr EnclaveObject() noexcept : enclave{}, id{} {}

  __declspec(dllexport) virtual ~EnclaveObject() noexcept;

  friend constexpr void swap(EnclaveObject &first,
                             EnclaveObject &second) noexcept {
    using std::swap;
    swap(first.enclave, second.enclave);
    swap(first.id, second.id);
  }

  EnclaveObject(EnclaveObject &&other) noexcept : EnclaveObject{} {
    swap(*this, other);
  }

  constexpr EnclaveObject &operator=(EnclaveObject &&other) noexcept {
    swap(*this, other);
    return *this;
  }
};

// 可以默认构造
static_assert(std::is_nothrow_default_constructible_v<EnclaveObject>);
// 可以析构
static_assert(std::is_nothrow_destructible_v<EnclaveObject> &&
              std::has_virtual_destructor_v<EnclaveObject>);
// 不可复制
static_assert(not std::is_copy_constructible_v<EnclaveObject> &&
              not std::is_copy_assignable_v<EnclaveObject>);
// 可以移动
static_assert(std::is_nothrow_move_constructible_v<EnclaveObject> &&
              std::is_nothrow_move_assignable_v<EnclaveObject> &&
              std::is_nothrow_swappable_v<EnclaveObject>);

class Decryptor : EnclaveObject {
  using EnclaveObject::EnclaveObject;

  friend DecryptingKey;

public:
  [[nodiscard]]
  std::vector<uint8_t> decrypt(uint8_t const *ciphertext, size_t len);

  [[nodiscard]]
  std::vector<uint8_t> finish();
};

class DecryptingKey : EnclaveObject {
  using EnclaveObject::EnclaveObject;

  friend EncryptingKey;

public:
  [[nodiscard]]
  Decryptor decrypt(std::vector<uint8_t> const &iv,
                    std::vector<uint8_t> const &tag);
};

struct EncryptFinish {
  std::vector<uint8_t> ciphertext;

  std::vector<uint8_t> tag;
};

class Encryptor : EnclaveObject {
  using EnclaveObject::EnclaveObject;

  friend EncryptingKey;

public:
  [[nodiscard]]
  std::vector<uint8_t> encrypt(uint8_t const *plaintext, size_t len);

  [[nodiscard]]
  EncryptFinish finish();
};

struct EncryptInit {
  std::vector<uint8_t> encrypted_key;

  std::vector<uint8_t> iv;

  Encryptor encryptor;

  DecryptingKey decrypt_key;
};

class EncryptingKey : EnclaveObject {
  using EnclaveObject::EnclaveObject;

public:
  [[nodiscard]]
  EncryptInit encrypt();
};

class SigningKey : EnclaveObject {
  using EnclaveObject::EnclaveObject;

public:
  std::vector<uint8_t> sign(uint8_t const *data, size_t len);
  std::vector<uint8_t> sign(std::string_view data);
};

class EnclaveKeyStore {
private:
  Enclave enclave;

public:
  EnclaveKeyStore() noexcept = default;

  constexpr EnclaveKeyStore(Enclave &&enclave) noexcept
      : enclave{std::move(enclave)} {}

  /**
   * 解析并导入 RSA 公钥, 可用于加密.
   * @param pem PEM 编码的 RSA 公钥.
   * @returns 新导入的 RSA 公钥的引用.
   */
  EncryptingKey load_rsa_public_key(std::string_view pem);

  /**
   * 解析并导入 RSA 私钥, 可用于签名.
   * @param blob 在 OpenSSL 后端上, 为 PEM 编码的 RSA 私钥; 在 BCrypt 后端上, 为
   * BCRYPT_RSAKEY_BLOB.
   * @returns 新导入的 RSA 私钥的引用.
   */
  SigningKey load_rsa_private_key(std::string_view blob);
};

// 可以默认构造
static_assert(std::is_nothrow_default_constructible_v<EnclaveKeyStore>);
// 可以析构
static_assert(std::is_nothrow_destructible_v<EnclaveKeyStore>);
// 不可复制
static_assert(not std::is_copy_constructible_v<EnclaveKeyStore> and
              not std::is_copy_assignable_v<EnclaveKeyStore>);
// 可以移动
static_assert(std::is_nothrow_move_constructible_v<EnclaveKeyStore> and
              std::is_nothrow_move_assignable_v<EnclaveKeyStore> and
              std::is_nothrow_swappable_v<EnclaveKeyStore>);

} // namespace detail
} // namespace jsc
