#pragma once

#include <cstddef> // For size_t
#include <cstdint>
#include <tuple>

#include <jsc/detail/consts.hpp> // IWYU pragma: export

namespace jsc {
namespace detail {

namespace vbs {

using routine_id_t = uint32_t;

using object_id_t = uint32_t;

struct free_object {
  static constexpr routine_id_t id = 0;
  static constexpr char const *name = "free_object";

  using signature = void(object_id_t id);
};

struct load_rsa_public_key {
  static constexpr routine_id_t id = 1;
  static constexpr char const *name = "load_rsa_public_key";

  using signature = object_id_t(char const *blob, size_t len);
};

struct load_rsa_private_key {
  static constexpr routine_id_t id = 2;
  static constexpr char const *name = "load_rsa_private_key";

  using signature = object_id_t(char const *blob, size_t len);
};

struct encrypt_init {
  static constexpr routine_id_t id = 3;
  static constexpr char const *name = "encrypt_init";

  /** @returns encryptor id */
  using signature = object_id_t(object_id_t rsa_key_id, uint8_t *enc_key,
                                size_t *enc_key_len, uint8_t *iv,
                                object_id_t *sym_key_id);
};

struct encrypt_update {
  static constexpr routine_id_t id = 4;
  static constexpr char const *name = "encrypt_update";

  using signature = void(object_id_t encryptor_id, uint8_t const *plaintext,
                         size_t plaintext_len, uint8_t *ciphertext,
                         size_t *ciphertext_len);
};

struct encrypt_finish {
  static constexpr routine_id_t id = 5;
  static constexpr char const *name = "encrypt_finish";

  using signature = void(object_id_t encryptor_id, uint8_t *ciphertext,
                         size_t *ciphertext_len, uint8_t *tag);
};

struct decrypt_init {
  static constexpr routine_id_t id = 6;
  static constexpr char const *name = "decrypt_init";

  using signature = object_id_t(object_id_t key_id, uint8_t const *iv,
                                uint8_t const *tag);
};

struct decrypt_update {
  static constexpr routine_id_t id = 7;
  static constexpr char const *name = "decrypt_update";

  using signature = void(object_id_t decryptor_id, uint8_t const *ciphertext,
                         size_t ciphertext_len, uint8_t *plaintext,
                         size_t *plaintext_len);
};

struct decrypt_finish {
  static constexpr routine_id_t id = 8;
  static constexpr char const *name = "decrypt_finish";

  using signature = void(object_id_t decryptor_id, uint8_t *plaintext,
                         size_t *plaintext_len);
};

struct rsa_sign {
  static constexpr routine_id_t id = 9;
  static constexpr char const *name = "rsa_sign";

  using signature = void(object_id_t key_id, uint8_t const *data, size_t len,
                         uint8_t *out, size_t *out_len);
};

static constexpr routine_id_t kRoutineCount = 10;

} // namespace vbs

template <typename Routine>
struct routine_traits;

template <typename Result, typename... Args>
struct routine_traits<Result(Args...)> {
  using args = std::tuple<Args...>;
  using result = Result;
};

template <typename Routine>
using routine_args_t =
    typename routine_traits<typename Routine::signature>::args;

template <typename Routine>
using routine_result_t =
    typename routine_traits<typename Routine::signature>::result;

template <typename Routine>
struct routine_args_with_error_t {
  routine_args_t<Routine> args;
  char *error;
  size_t *error_len;
};

} // namespace detail
} // namespace jsc
